unit Oscillograph01;

interface

uses Windows, Messages, Classes, Controls, SysUtils, Graphics, Dialogs,
     StdCtrls, ExtCtrls, Menus;

// ====================================================================
//                  TItem1 ( 1.0)
//       
// ====================================================================
type TItem1 = class(TObject)
protected
  fNext   : TItem1;               //    
  fData   : extended;             // ,    
end;
// ====================================================================
//                   TQueue1 ( 1.0)
//   ,     FIFO
// ====================================================================
type TQueue1 = class(TItem1)
private
  fHead     : TItem1;            //    
  fTail     : TItem1;            //     
  fCurrent  : TItem1;            //    
  fCount    : integer;           //    
  fMaxCount : integer;           //    
  procedure   InsertToTail(RqData : extended);
  procedure   CutFromHead();
  procedure   SetQueueMaxCount(RqCount : integer);
public
  constructor Create();
  procedure   AddData (RqData : extended);
  function    SetCurrentItem(Cmd : char) : boolean;
  function    GetCurrentData () : extended;
  procedure   ClearQueue();
  procedure   Free();
  property    QueueMaxCount : integer read fMaxCount write SetQueueMaxCount;
end;
// ====================================================================
//                TOscillograph
//            
// ====================================================================
type TOscillograph = class(TObject)
private
   // -----------------------------
   fTitle      : string;       //  
   // -----------------------------
   fPanelRect  : Trect;        //   
   fPanel      : TPanel;       //  
   fDisplay    : TImage;       // Image  
   fPM         : TPopupMenu;   // PopUpMenu  
   fBeam1      : TQueue1;      // FIFO   
   fLbT        : TLabel;       // Label  
   fLbV        : TLabel;       // Label  
   // -----------------------------
   //  
   fYMes       : string;       //    
   fYMax       : extended;     // .    
   fXMax       : integer;      // .    
   // -----------------------------
   //  
   fDigit      : boolean;      //   
   fdYShow     : boolean;      //    
   fYPrev      : extended;     //   
   fXCount     : integer;      //    
   // -----------------------------
   //  
   fGridY      : extended;     //     Y
   fGridX      : integer;      //     X
   // -----------------------------
   //  
   fBeamColor  : TColor;       //  
   fFonColor   : TColor;       //   
   fGridColor  : TColor;       //    
   // -----------------------------
   //   
   fAlarmOn    : boolean;      //    
   fY1Alarm    : extended;     //    1
   fY2Alarm    : extended;     //    2
   fc1Alarm    : TColor;       //     1
   fc2Alarm    : TColor;       //     2
   // -----------------------------
   //  property
   procedure SetTitle(RqTitle : string);
   procedure SetYMax (RqYMax  : extended);
   procedure SetXMax (RqXMax  : integer);
   // -----------------------------
   //    PopupMenu
   procedure MenuClick(Sender : TObject);
   // -----------------------------
   //      
   procedure ShowGrids();
   //  
   procedure ImageClear(RqColor : TColor);
   //      X
   function  CalcX() : integer;
   //      Y
   function  CalcY(RqData : extended) : integer;
   //    
   procedure ShowFirstPoint(RqData : extended);
   //    
   procedure ShowNextPoint(RqData : extended);
public
   // -----------------------------
   //        
   constructor Create(RqParent : TWinControl; RqPanelRect : TRect);
   procedure   Free;
   // -----------------------------
   //    
   procedure AddNewData (RqData : extended);
   //   
   procedure ShowBeam();
   // -----------------------------
   //  
   property Title      : string   read fTitle     write SetTitle;
   //   
   property PanelRect  : Trect    read fPanelRect;
    //    
   property  YMes      : string   read fYMes     write fYMes;
   //   
   property YMax       : extended read fYMax      write SetYMax;
   //      X
   property XMax       : integer  read fXMax      write SetXMax;
   //   
   property Digit      : boolean  read fDigit     write fDigit;
   //   
   property BeamColor  : TColor   read fBeamColor write fBeamColor;
   //   
   property FonColor   : TColor   read fFonColor  write fFonColor;
   //   
   property GridColor  : TColor   read fGridColor write fGridColor;
   //     Y (   )
   property GridY      : extended read fGridY     write fGridY;
   //     X ( )
   property GridX      : integer  read fGridX     write fGridX;
   //    (   )
   property Y1Alarm    : extended read fY1Alarm   write fY1Alarm;
   //    (   )
   property Y2Alarm    : extended read fY2Alarm   write fY2Alarm;
   //     
   property Alarm1Color  : TColor read fc1Alarm   write fc1Alarm;
   //     
   property Alarm2Color  : TColor read fc2Alarm   write fc2Alarm;

end;


// ====================================================================
// ====================================================================
implementation
// ====================================================================
// ====================================================================


// ====================================================================
//                   TQueue1  ( 1.0)
//   ,     FIFO
// ====================================================================
// --------------------------------------------------------------------
//   ,        
procedure TQueue1.InsertToTail(RqData : extended);
var  wp : TItem1;
begin
   wp := TItem1.Create;
   wp.fNext := nil;
   wp.fData := RqData;
   if not Assigned(fTail)
   then begin  //     
      fTail    := wp;
      fHead    := fTail;
      fCurrent := fTail;
      fCount   := 1;
   end
   else begin  //     
      fTail.fNext := wp;
      fTail := wp;
      fCount := fCount + 1;
   end;
end;
// --------------------------------------------------------------------
//     
procedure TQueue1.CutFromHead();
var  wp  :  TItem1;
begin
   if not Assigned(fHead)
   then begin
      //  
      fTail    := nil;
      fCurrent := nil;
      fCount   := 0;
   end
   else begin
      //     
      wp := fHead;
      if not Assigned(wp.fNext)
      then begin   //       
          fHead    := nil;
          fTail    := nil;
          fCurrent := nil;
          fCount   := 0;
      end
      else begin   //    
        fHead := wp.fNext;
        //     
        if fCurrent = wp then fCurrent := fHead;
        fCount := fCount - 1;
        if fCount < 1 then ShowMessage(' Queue');
      end;
      //   
      wp.Free;
   end;
end;
// --------------------------------------------------------------------
//  
constructor TQueue1.Create();
begin
   inherited Create;
   fHead     := nil;   //  
   fTail     := nil;   //  
   fCurrent  := nil;   //  
   fCount    := 0;     //  
   fMaxCount := 10;    //   
end;
// --------------------------------------------------------------------
//     
procedure TQueue1.AddData (RqData : extended);
begin
   InsertToTail(RqData);
   if fCount > fMaxCount then CutFromHead();
end;
// --------------------------------------------------------------------
//    
function TQueue1.SetCurrentItem(Cmd : char) : boolean;
begin
    Result   := False;
    if not Assigned(fHead)
    then begin
        fCurrent := nil;
    end
    else begin
       //   
       case Cmd of
       'H' : begin    //     () 
                fCurrent := fHead;
                Result := True;
             end;
       'N' : begin   //      
                if Assigned(fCurrent)
                then begin
                   fCurrent := fCurrent.fNext;
                   if Assigned(fCurrent) then Result := True;
                end
                else fCurrent := nil;
            end;
       end;
    end;
end;
// --------------------------------------------------------------------
//      
function  TQueue1.GetCurrentData () : extended;
begin
   Result := 0;
   if Assigned(fCurrent) then Result := fCurrent.fData;
end;
// --------------------------------------------------------------------
//     
procedure TQueue1.ClearQueue();
begin
   while Assigned(fHead) do CutFromHead();
end;
// --------------------------------------------------------------------
//        
procedure TQueue1.SetQueueMaxCount(RqCount : integer);
begin
   if RqCount > 0
   then begin
      ClearQueue();
      fMaxCount := RqCount;
   end;
end;
// --------------------------------------------------------------------
//  
procedure  TQueue1.Free();
begin
  ClearQueue();
  inherited Free();
end;

// ====================================================================
//                TOscillograph
//            
// ====================================================================
const MinPanelW = 200;  // .   
      MinPanelH = 120;  // .   
// --------------------------------------------------------------------
//   
constructor TOscillograph.Create(RqParent : TWinControl; RqPanelRect : TRect);
var MenuItems : array of TMenuItem;
begin
   inherited Create;
   fPanelRect := RqPanelRect;
   with RqPanelRect do
   begin
      if Left < 0 then fPanelRect.Left := 0;
      if Top < 0 then  fPanelRect.Top  := 0;
      if Right - fPanelRect.Left < MinPanelW
      then fPanelRect.Right := fPanelRect.Left + MinPanelW;
      if Bottom - fPanelRect.Top < MinPanelH
      then fPanelRect.Bottom := fPanelRect.Top + MinPanelH;
   end;
   // -----------------------------
   //   
   fPanel := TPanel.Create(RqParent);
   fPanel.Parent := RqParent;
   fPanel.Visible := False;
   with fPanelRect do fPanel.SetBounds(Left,Top,Right-Left,Bottom-Top);
   // -----------------------------
   //  Image  
   fDisplay  := TImage.Create(fPanel);
   fDisplay.Parent := fPanel;
   fDisplay.SetBounds(2,20,fPanel.Width-4,fPanel.Height-40);
   // -----------------------------
   //  Label    
   fLbV := TLabel.Create(fPanel);
   fLbV.Parent := fPanel;
   fLbV.SetBounds(2,fPanel.Height-18,fPanel.Width-4,16);
   fLbV.AutoSize := False;
   // -----------------------------
   //  Label   
   fLbT := TLabel.Create(fPanel);
   fLbT.Parent := fPanel;
   fLbT.SetBounds(2,3,fPanel.Width-4,16);
   fLbT.AutoSize := False;
   fLbT.Alignment := taCenter;
   // -----------------------------
   fPanel.Visible := True;
   // -----------------------------
   //    PopupMenu
   SetLength(MenuItems, 3);
   MenuItems[0]:= NewItem(' ', TextToShortCut(''),
                          True, True, MenuClick, 0, 'ITEM1');
   MenuItems[0].Tag := 1;
   MenuItems[1]:= NewItem(' ', TextToShortCut(''),
                          True, True, MenuClick, 0, 'ITEM2');
   MenuItems[1].Tag := 2;
   MenuItems[2]:= NewItem('', TextToShortCut(''),
                          False, True, MenuClick, 0, 'ITEM3');
   MenuItems[2].Tag := 3;
   fPM := NewPopupMenu(fDisplay, 'Menu',
                                paLeft, True, MenuItems);
   SetLength(MenuItems,0);
   fDisplay.PopupMenu := fPM;
   // -----------------------------
   //  FIFO    
   fBeam1  := TQueue1.Create;
   fXMax   := 50;
   fBeam1.QueueMaxCount := fXMax;
   // -----------------------------
   //   
   fYMax  := 100;
   // -----------------------------
   //    
   fGridY := fYMax / 5;
   fGridX := fXMax div 5;
   fDigit := True;
   // -----------------------------
   //    
   fBeamColor  := clLime;  // RGB(64,64,128);
   fFonColor   := RGB(32,32,32); // clBlack;
   fGridColor  := clGray;
   ImageClear(fFonColor);
   // -----------------------------
   //  
   fAlarmOn    := True;             //    
   fY1Alarm    := fYMax * 0.7;      //    1
   fY2Alarm    := fYMax * 0.9;      //    2
   fc1Alarm    := clYellow; // RGB(180,180,0);   //     1
   fc2Alarm    := clRed;    // RGB(200,0,0);     //     1
end;
// --------------------------------------------------------------------
procedure TOscillograph.Free;
begin
   if Assigned(fBeam1)   then fBeam1.Free;
   if Assigned(fPM)      then fPM.Free;
   if Assigned(fLbV)     then fLbV.Free;
   if Assigned(fLbT)     then fLbT.Free;
   if Assigned(fDisplay) then fDisplay.Free;
   if Assigned(fPanel)   then fPanel.Free;
   inherited Free;
end;
// --------------------------------------------------------------------
// --------------------------------------------------------------------
//    
procedure TOscillograph.MenuClick(Sender : TObject);
var Item       : TMenuItem;
begin
   Item  :=  TMenuItem(Sender);
   case Item.Tag of
   1 : begin  //  
          if not Item.Checked
          then begin
             fAlarmOn := True;
             Item.Checked := True;
          end
          else begin
             fAlarmOn := False;
             Item.Checked := False;
          end;
       end;
   2 : begin //  
          if not Item.Checked
          then begin
             fDigit := True;
             Item.Checked := True;
          end
          else begin
             fDigit := False;
             Item.Checked := False;
          end;
       end;
   3 : begin  // 
          if not Item.Checked
          then begin
             fdYShow := True;
             Item.Checked := True;
          end
          else begin
             fdYShow := False;
             Item.Checked := False;
          end;
       end;
   else begin end;
   end;
end;
// --------------------------------------------------------------------
procedure TOscillograph.SetTitle(RqTitle : string);
begin
   if Assigned(fLbT) then fLbT.Caption := RqTitle;
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.SetYMax(RqYMax : extended);
begin
   if RqYMax < 1e-10
   then fYMax := 1e-10
   else fYMax := RqYMax;
   ImageClear(fFonColor);
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.SetXMax(RqXMax : integer);
begin
  if RqXMax < 8
  then fBeam1.QueueMaxCount := 8
  else fBeam1.QueueMaxCount := RqXMax;
  ImageClear(fFonColor);
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.ShowGrids();
var wY : extended;
    wX : integer;
begin
    with fDisplay.Canvas
    do begin
       //    Y
       //  
       Pen.Color := (not fFonColor) and $00FFFFFF;
       MoveTo(0, CalcY(0));
       LineTo(fDisplay.Width,CalcY(0));
       //  
       Pen.Color := fGridColor;
       //    
       wY := - fGridY;
       repeat
          MoveTo(0, CalcY(wY));
          LineTo(fDisplay.Width,CalcY(wY));
          wY := wY - fGridY;
       until (wY < - fYMax);
       //    
       wY := fGridY;
       repeat
          MoveTo(0, CalcY(wY));
          LineTo(fDisplay.Width,CalcY(wY));
          wY := wY + fGridY;
       until (wY > fYMax);
       //    X
       wX := 0;
       repeat
          MoveTo(wX, 0,);
          LineTo(wX, fDisplay.Height);
          wX := wX + fGridX;
       until (wX > fDisplay.Width);
    end;
end;
// --------------------------------------------------------------------
//     
procedure TOscillograph.ImageClear(RqColor : TColor);
begin
  with fDisplay.Canvas do
  begin
    Brush.Color := RqColor;
    Brush.Style := bsSolid;
    FillRect(Rect(0,0, fDisplay.Width, fDisplay.Height));
  end;
  fXCount := 0;
end;
// --------------------------------------------------------------------
//  X -   
function TOscillograph.CalcX() : integer;
begin
  Result := Trunc((fDisplay.Width / fBeam1.QueueMaxCount) * fXCount);
end;
// --------------------------------------------------------------------
//  Y -   
function TOscillograph.CalcY(RqData : extended) : integer;
begin
  Result := (fDisplay.Height div 2);
  Result := Result - Trunc((Result / fYMax) * RqData);
end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.ShowFirstPoint(RqData : extended);
begin
   //   
  ImageClear(fFonColor);
  ShowGrids();
  //   
  fDisplay.Canvas.MoveTo(CalcX(),CalcY(RqData));
  fYPrev  := RqData;
  fXCount := fXCount + 1;
end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.ShowNextPoint(RqData : extended);
var fwBeamColor : TColor;
begin
   fwBeamColor := fBeamColor;
   if fAlarmOn
   then begin
     //     
     if Abs(RqData) >= fY1Alarm then fwBeamColor := fc1Alarm;
     if Abs(RqData) >= fY2Alarm then fwBeamColor := fc2Alarm;
   end;
   fDisplay.Canvas.Pen.Color := fwBeamColor;
   if fDigit
   then fDisplay.Canvas.LineTo(CalcX(),CalcY(fYPrev));
   fDisplay.Canvas.LineTo(CalcX(),CalcY(RqData));
   fYPrev  := RqData;
   fXCount := fXCount + 1;
end;
// --------------------------------------------------------------------
//  PUBLIC 
// --------------------------------------------------------------------
//     
procedure TOscillograph.AddNewData(RqData : extended);
begin
   if Assigned (fBeam1) then fBeam1.AddData(RqData);
   if Assigned (fLbV)
   then begin
     if not fdYShow
     then fLbV.Caption := '   : '
                       + Format('%6.3f',[RqData]) + ' ' + YMes
     else fLbV.Caption := '   : '
                       + Format('%6.3f',[RqData-fYPrev]) + ' ' + YMes;
   end;

end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.ShowBeam();
begin
   if Assigned (fBeam1)
   then begin
      if fBeam1.SetCurrentItem('H')
      then begin
         ShowFirstPoint(fBeam1.GetCurrentData());
         while fBeam1.SetCurrentItem('N')
         do ShowNextPoint(fBeam1.GetCurrentData());
      end;
   end;
end;
// ====================================================================
// ====================================================================

end.
