unit DTF03;
// ===========================================================================
(*
          
          .
    :
     // -----------------------------------------------------
     //    / ( property Pn)
     DTF.RqDegree := False;           // 
     //   
     DTF.CodMet := 0;                 // 0 -  , 1 - 
     //   
     DTF.XB := XB;                    //   X
     DTF.XE := XE;                    //    X
     //  DTF    ArrY
     DTF.RunDTF03(ArrY);              // var ArrY : array of extended
     // -----------------------------------------------------
     //     property:
     // ------------------------------
     //    
     DTF.An[Indx] - .  Sin  
     DTF.Bn[Indx] - .  Cos  
     // ------------------------------
     //      SUM( Cn * Sin(n * Omega * X + Pn))
     // Pn -  
     // Omega -   2 * Pi / (XE - XB)
     DTF.Cn[Indx] -    Sin  
     DTF.Pn[Indx] -  (/)   Sin  
     // -----------------------------------------------------
          X  .  :
     Y := DTF.CalcFuncTDF(X);
*)
// ===========================================================================
interface

uses Dialogs, DateUtils, Math, SysUtils, StdCtrls;

// ===========================================================================
//   
// ===========================================================================
type TSnCs  = record
   Sn  : extended;  //  SIN
   Cs  : extended;  //  COS
end;

type TDTF03 = class(TObject)
  private
     fMNumPnt  : integer;            //     
     fMNumGrm  : integer;            //   
     fMIndGrm  : integer;            //   
     fDegree   : boolean;            //      
     // 
     fXB       : extended;           //   
     fXE       : extended;           //    
     fStepX    : extended;
     fOmega    : extended;
     // ------------------
     fCodMet   : byte;               //     (0,1)
     // ------------------
     //  
     fBSnCs    : TSnCs;
     fESnCs    : TSnCs;
     //       
     fQuadTAB  : array of record
                    //    SIN
                    SS2  : extended; //    : 0  2
                    SS1  : extended; //    : 1
                    SSM  : extended; //  -   1..4
                    SSUM : extended; //   
                    //    COS
                    CS2  : extended; //    : 0  2
                    CS1  : extended; //    : 1
                    CSM  : extended; //  -   1..4
                    CSUM : extended; //   
                    //       
                    An   : extended;
                    Bn   : extended;
                    Cn   : extended;
                    Pn   : extended;
                 end;
     //    
     procedure ClearQuadTAB();
     //    fQuadTAB =    
     function ReSetQuadTAB(RqArr : array of extended) : boolean;
     //  fOmega  fStepX
     procedure CalcQmegaAndStepX(RqArr : array of extended);
     //  Sin(X)  Cos(X)       
     function GetSnCs(RqIndG, RqIndX : integer) : TSnCs;
     //      
     function TrapeziumQuadrature(RqArr : array of extended) : boolean;
     //      
     function SimpsonQuadrature(RqArr : array of extended) : boolean;
     //   An, Bn     
     procedure CalcAnBn(RqArr : array of extended; IndG : integer);
     //   An, Bn     Cn, Pn
     procedure AnBnToCnPn(IndG : integer);
     //      
     procedure CalcQuadTAB(RqArr : array of extended);
     //  property
     function GetAn(Ind : word) : extended;
     function GetBn(Ind : word) : extended;
     function GetCn(Ind : word) : extended;
     function GetPn(Ind : word) : extended;
     //    
     procedure SetXB(RqXB : extended);
     //    
     procedure SetXE(RqXE : extended);
  public
     procedure Free();
     //    
     procedure RunDTF03(RqArr : array of extended);
     //      ,  
     // function CalcFuncTDF (RqInd : word) : extended; overload;
     //      ,  
     function CalcFuncTDF(RqX : extended) : extended; // overload;
     //   
     //     
     procedure LookDTF03(RqRep : TMemo);
     //   
     property XB : extended read fXB write SetXB;
     property XE : extended read fXE write SetXE;
     property StepX : extended read fStepX;
     //     
     property An[Ind : word] : extended read GetAn;
     property Bn[Ind : word] : extended read GetBn;
     property Cn[Ind : word] : extended read GetCn;
     property Pn[Ind : word] : extended read GetPn;
     //  /       / 
     property RqDegree  : boolean read fDegree write fDegree;
     //   
     property MaxIndGgm : integer read fMIndGrm;
     //     0-, 1 - 
     property CodMet    : byte read fCodMet write fCodMet;
end;

implementation
// ---------------------------------------------------------------------------
const MinNumPnt = 6;     //      

// ---------------------------------------------------------------------------
// 28.02.2014
procedure TDTF03.Free();
begin
   SetLength (fQuadTAB, 0);
   inherited Free();
end;

// ---------------------------------------------------------------------------
// 23.03.2014
//   PROPERTY
// --------------------------------------------------------------------
function TDTF03.GetAn(Ind : word) : extended;
begin
    if Ind <= fMIndGrm then Result := fQuadTAB[Ind].An
    else Result := 0;
end;
// --------------------------------------------------------------------
function TDTF03.GetBn(Ind : word) : extended;
begin
    if Ind <= fMIndGrm then Result := fQuadTAB[Ind].Bn
    else Result := 0;
end;
// --------------------------------------------------------------------
function TDTF03.GetCn(Ind : word) : extended;
begin
    if Ind <= fMIndGrm then Result := fQuadTAB[Ind].Cn
    else Result := 0;
end;
// --------------------------------------------------------------------
function TDTF03.GetPn(Ind : word) : extended;
begin
  if Ind <= fMIndGrm
  then begin
    if not fDegree
    then Result := fQuadTAB[Ind].Pn
    else Result := 180 * fQuadTAB[Ind].Pn /Pi;
  end
  else Result := 0;
end;
// --------------------------------------------------------------------
//    
procedure TDTF03.SetXB(RqXB : extended);
begin
   if (RqXB = fXB) then Exit;
   if (RqXB >= fXE)
   then begin
      MessageDlg('     ' +  #13#10
               + '    ...',
          mtError, [mbOk], 0);
      Exit;
   end;
   fXB := RqXB;
end;
// --------------------------------------------------------------------
//    
procedure TDTF03.SetXE(RqXE : extended);
begin
   if (RqXE = fXE) then Exit;
   if (RqXE <= fXB)
   then begin
      MessageDlg('     ' +  #13#10
               + '    ...',
          mtError, [mbOk], 0);
      Exit;
   end;
   fXE := RqXE;
end;
// ---------------------------------------------------------------------------
// 01.03.2014
//    
procedure TDTF03.ClearQuadTAB();
var Ind : integer;
begin
   if Length(fQuadTAB) > 0
   then begin
     for Ind := Low(fQuadTAB) to High(fQuadTAB)
     do FillChar(fQuadTAB[Ind], SizeOF(fQuadTAB[Ind]), #0);
   end;
end;
// ---------------------------------------------------------------------------
// 01.03.2014
//    fQuadTAB =    
function TDTF03.ReSetQuadTAB(RqArr : array of extended) : boolean;
// Length(RqArr) -     RqArr;
// High(RqArr)   -    =     RqArr;
// High(RqArr) div 2 -   ;
// High(RqArr) mod 2 -      = 0  1;
//        = 4;
begin
   try
      //  .        
      if (fMIndGrm <> High(RqArr) div 4)
      then begin
         //       = High(fQuadTAB)
         fMIndGrm := High(RqArr) div 4;
         //       = Length(fQuadTAB)
         fMNumGrm := 1 + fMIndGrm;
         //     
         SetLength (fQuadTAB, fMNumGrm);
      end;
      //    (  )
      ClearQuadTAB();
      Result := True;
   except
      Result := False;
   end;
end;
// ---------------------------------------------------------------------------
// 30.08.2017
//  fOmega  fStepX
procedure TDTF03.CalcQmegaAndStepX(RqArr : array of extended);
const TwoPi = 2 * Pi;
begin
   fStepX  := (fXE - fXB) / High(RqArr);
   fOmega  :=  TwoPi / (fXE - fXB);
end;
// ---------------------------------------------------------------------------
// 30.08.2017
//  Sin(X)  Cos(X)       
function TDTF03.GetSnCs(RqIndG, RqIndX : integer) : TSnCs;
begin
   SinCos(RqIndG * fOmega * (fXB + fStepX * RqIndX),
          Result.Sn, Result.Cs);
end;

// ---------------------------------------------------------------------------
// 30.08.2017
//      
function TDTF03.TrapeziumQuadrature(RqArr : array of extended) : boolean;
var   IndX               : integer;  //     
      IndG               : integer;  //   
      Sum0, Sum1         : extended; //  
begin
  //    
  ClearQuadTAB();
  //  fOmega  fStepX
  CalcQmegaAndStepX(RqArr);
  //       
  for IndG := 0 to High(fQuadTAB) do
  begin
     // Sin  Cos     RqArr
     fBSnCs := GetSnCs(IndG, Low(RqArr));
     //       
     for IndX := Low(RqArr) to High(RqArr) - 1
     do begin
          // Sin  Cos     RqArr
          fESnCs := GetSnCs(IndG, IndX + 1);
          //     
          Sum0 := RqArr[IndX]     * fBSnCs.Sn;
          Sum1 := RqArr[IndX + 1] * fESnCs.Sn;
          fQuadTAB[IndG].SSUM := fQuadTAB[IndG].SSUM + Sum0 + Sum1;
          //     
          Sum0 := RqArr[IndX]     * fBSnCs.Cs;
          Sum1 := RqArr[IndX + 1] * fESnCs.Cs;
          fQuadTAB[IndG].CSUM := fQuadTAB[IndG].CSUM + Sum0 + Sum1;
          // Sin  Cos      RqArr
          fBSnCs := fESnCs;
       end;
      //     
      fQuadTAB[IndG].SSUM := fQuadTAB[IndG].SSUM / 2;
      fQuadTAB[IndG].CSUM := fQuadTAB[IndG].CSUM / 2;
  end;
  Result := True;
end;
// ---------------------------------------------------------------------------
// 30.08.2017
//      
function TDTF03.SimpsonQuadrature(RqArr : array of extended) : boolean;
var   StartIndX  : integer;  //     
      IndX       : integer;  //     
      IndG       : integer;  //   
      //----------
      //   Sin  Cos    
      w1SnCs     : TSnCs;
      w2SnCs     : TSnCs;
      w3SnCs     : TSnCs;
      w4SnCs     : TSnCs;
      //----------
      Sum0, Sum1, Sum2 : extended; //    
      Sum3, Sum4       : extended; //    - 
begin
  Result := False;
  //    
  ClearQuadTAB();
  //     
  if High(RqArr) < 5 then Exit;
  //  fOmega  fStepX
  CalcQmegaAndStepX(RqArr);
  // ----------------------------------------------------
  //       - 
  if (Length(RqArr) mod 2) = 0       //   
  then begin
      //   
      for IndG := 0 to High(fQuadTAB)
      do begin
          // ---------------------------------------
          //    
          //     0, 1, 2
          IndX := Low(RqArr);
          // Sin  Cos    
          fBSnCs := GetSnCs(IndG, IndX);
          // Sin  Cos    
          w1SnCs := GetSnCs(IndG, IndX + 1);
          // Sin  Cos   ()  
          fESnCs := GetSnCs(IndG, IndX + 2);
          //     
          Sum0 := RqArr[IndX]     * fBSnCs.Sn;
          Sum1 := RqArr[IndX + 1] * w1SnCs.Sn;
          Sum2 := RqArr[IndX + 2] * fESnCs.Sn;
          fQuadTAB[IndG].SS2 := fQuadTAB[IndG].SS2 + Sum0 + Sum2;
          fQuadTAB[IndG].SS1 := fQuadTAB[IndG].SS1 + Sum1;
          //     
          Sum0 := RqArr[IndX]     * fBSnCs.Cs;
          Sum1 := RqArr[IndX + 1] * w1SnCs.Cs;
          Sum2 := RqArr[IndX + 2] * fESnCs.Cs;
          fQuadTAB[IndG].CS2 := fQuadTAB[IndG].CS2 + Sum0 + Sum2;
          fQuadTAB[IndG].CS1 := fQuadTAB[IndG].CS1 + Sum1;
          // ---------------------------------------
          //   - 
          //     1, 2, 3, 4.
          // Sin  Cos    - 
          // w1SnCs
          // Sin  Cos    - 
          w2SnCs :=  fESnCs;
          // Sin  Cos    - 
          w3SnCs := GetSnCs(IndG, IndX + 3);
          // Sin  Cos    - 
          w4SnCs := GetSnCs(IndG, IndX + 4);
          //     
          Sum1 := RqArr[IndX + 1] * w1SnCs.Sn;
          Sum2 := RqArr[IndX + 2] * w2SnCs.Sn;
          Sum3 := RqArr[IndX + 3] * w3SnCs.Sn;
          Sum4 := RqArr[IndX + 4] * w4SnCs.Sn;
          fQuadTAB[IndG].SSM := ((Sum2 + Sum3) * 13 - Sum1 - Sum4) / 24;
          //     
          Sum1 := RqArr[IndX + 1] * w1SnCs.Cs;
          Sum2 := RqArr[IndX + 2] * w2SnCs.Cs;
          Sum3 := RqArr[IndX + 3] * w3SnCs.Cs;
          Sum4 := RqArr[IndX + 4] * w4SnCs.Cs;
          fQuadTAB[IndG].CSM := ((Sum2 + Sum3) * 13 - Sum1 - Sum4) / 24;
          // ---------------------------------------
      end;
      //        
      StartIndX := 3;
  end
  else begin
      //        
      StartIndX := 0;
  end;
  // ----------------------------------------------------
  //     
  for IndG := 0 to High(fQuadTAB)
  do begin
     IndX := StartIndX;
     // Sin  Cos    
     fBSnCs := GetSnCs(IndG, IndX);
     repeat
          // Sin  Cos    
          w2SnCs := GetSnCs(IndG, IndX + 1);
          // Sin  Cos    
          fESnCs := GetSnCs(IndG, IndX + 2);
          //     
          Sum0 := RqArr[IndX]     * fBSnCs.Sn;
          Sum1 := RqArr[IndX + 1] * w2SnCs.Sn;
          Sum2 := RqArr[IndX + 2] * fESnCs.Sn;
          fQuadTAB[IndG].SS2 := fQuadTAB[IndG].SS2 + Sum0 + Sum2;
          fQuadTAB[IndG].SS1 := fQuadTAB[IndG].SS1 + Sum1;
          //     
          Sum0 := RqArr[IndX]     * fBSnCs.Cs;
          Sum1 := RqArr[IndX + 1] * w2SnCs.Cs;
          Sum2 := RqArr[IndX + 2] * fESnCs.Cs;
          fQuadTAB[IndG].CS2 := fQuadTAB[IndG].CS2 + Sum0 + Sum2;
          fQuadTAB[IndG].CS1 := fQuadTAB[IndG].CS1 + Sum1;
          // Sin  Cos     
          fBSnCs := fESnCs;
          //    
          IndX := IndX + 2;

     until (IndX >= High(RqArr));
     //       
     fQuadTAB[IndG].SSUM := (4 * fQuadTAB[IndG].SS1 + fQuadTAB[IndG].SS2)/3;
     fQuadTAB[IndG].CSUM := (4 * fQuadTAB[IndG].CS1 + fQuadTAB[IndG].CS2)/3;
     //  - 
     fQuadTAB[IndG].SSUM := fQuadTAB[IndG].SSUM + fQuadTAB[IndG].SSM;
     fQuadTAB[IndG].CSUM := fQuadTAB[IndG].CSUM + fQuadTAB[IndG].CSM;
  end;
  Result := True;
end;
// ---------------------------------------------------------------------------
// 14.03.2014
//   An, Bn     
procedure TDTF03.CalcAnBn(RqArr : array of extended; IndG : integer);
var h : extended;
begin
  h := 1/High(RqArr);
  if IndG = 0
  then begin
     fQuadTAB[IndG].An := 0;
     fQuadTAB[IndG].Bn := fQuadTAB[IndG].CSUM * h;
  end
  else begin
     fQuadTAB[IndG].An := 2 * fQuadTAB[IndG].SSUM * h;
     fQuadTAB[IndG].Bn := 2 * fQuadTAB[IndG].CSUM * h;
  end;
end;
// ---------------------------------------------------------------------------
// 14.03.2014
//   An, Bn     Cn, Pn
procedure TDTF03.AnBnToCnPn(IndG : integer);
const Noise = 1E-12;              //   
begin
   //   Cn   ( )
   fQuadTAB[IndG].Cn := Sqrt(fQuadTAB[IndG].An * fQuadTAB[IndG].An
                           + fQuadTAB[IndG].Bn * fQuadTAB[IndG].Bn);
   //   Pn   ( )
   if fQuadTAB[IndG].Cn > Noise
   then begin
      // I  (  ARCSIN   Math)
      fQuadTAB[IndG].Pn := Abs(ARCSIN(fQuadTAB[IndG].Bn / fQuadTAB[IndG].Cn));
      //  
      if fQuadTAB[IndG].Bn >= 0
      then begin
         // I  II    ( Bn >= 0 )
         if fQuadTAB[IndG].An < 0
         then fQuadTAB[IndG].Pn := Pi - fQuadTAB[IndG].Pn;     // II 
      end
      else begin
         // III  IV  ( Bn < 0)
         if fQuadTAB[IndG].An < 0
         then fQuadTAB[IndG].Pn := fQuadTAB[IndG].Pn + Pi      // III 
         else fQuadTAB[IndG].Pn := 2 * Pi - fQuadTAB[IndG].Pn; // IV  
      end;
   end
   else begin
     //   
     fQuadTAB[IndG].Cn := 0;
     fQuadTAB[IndG].Pn := 0;
   end;
end;
// ---------------------------------------------------------------------------
// 29.05.2016
//      
procedure TDTF03.CalcQuadTAB(RqArr : array of extended);
var   IndG : integer;             //   
begin

  //    
  if (Length(RqArr) < 6) then Exit;
  if not(Low(RqArr) = 0) then Exit;

  //      
  case fCodMet of
     0 : TrapeziumQuadrature(RqArr);
     1 : SimpsonQuadrature(RqArr);
  end;
  //     
  for IndG := Low(fQuadTAB) to High(fQuadTAB)
  do begin
     //       An  Bn
     CalcAnBn(RqArr, IndG);
     //       Cn  Pn
     AnBnToCnPn(IndG);
  end; // of for IndG
end;  // of procedure CalcQuadTAB
// --------------------------------------------------------------------
// 30.08.2017
//      ,  
function TDTF03.CalcFuncTDF(RqX : extended) : extended;
const cEps = 1e-10;
var wIndG   : integer;
    wXToPhs : extended;
    wPhase  : extended;
    wY      : extended;
begin
   Result := 0;
   // 
   if Length(fQuadTAB) <= 0 then Exit;
   if (fXE < fXB) then Exit;
   if Abs(fXE - fXB) < cEps then fXE := fXB + cEps;
   //  
   wXToPhs := 2 * Pi /(fXE - fXB);
   //    
   wY := 0;
   for wIndG := Low(fQuadTAB) to High(fQuadTAB)
   do begin
      wPhase  := wXToPhs * wIndG * RqX + fQuadTAB[wIndG].Pn;
      if wIndG = Low(fQuadTAB)
      then wY := fQuadTAB[wIndG].Cn
      else wY := wY + fQuadTAB[wIndG].Cn * Sin(wPhase);
   end;
   Result := wY;
end;
// ---------------------------------------------------------------------------
// 02.03.2014
//  
procedure TDTF03.LookDTF03(RqRep : TMemo);
var Ind : integer;
    WStr : string;
begin
    RqRep.Clear;
    for Ind := 0 to High(fQuadTAB)
    do begin
      WStr := 'SSn' + IntToStr(Ind) + '=' + FloatToStr(fQuadTAB[Ind].SSUM);
      RqRep.Lines.Add(WStr);
      WStr := 'MSn' + IntToStr(Ind) + '=' + FloatToStr(fQuadTAB[Ind].SSM);
      RqRep.Lines.Add(WStr);

      WStr := 'SCs' + IntToStr(Ind) + '=' + FloatToStr(fQuadTAB[Ind].CSUM);
      RqRep.Lines.Add(WStr);
      WStr := 'MCs' + IntToStr(Ind) + '=' + FloatToStr(fQuadTAB[Ind].CSM);
      RqRep.Lines.Add(WStr);

      RqRep.Lines.Add('');
    end;
end;
// ---------------------------------------------------------------------------
// 30.08.2017
//    
procedure TDTF03.RunDTF03(RqArr : array of extended);
begin
   // ------------------------------------------
   //    
   if (Length(RqArr) < MinNumPnt) then Exit;
   if not(Low(RqArr) = 0) then Exit;
   //      
   fMNumPnt := Length(RqArr);
   //   
   if (fXE <= fXB)
   then begin
      MessageDlg('     ' +  #13#10
               + '    ...',
          mtError, [mbOk], 0);
      Exit;
   end;
   // ------------------------------------------
   // 
   //       DFT
   if not ReSetQuadTAB(RqArr) then Exit;
   //      
   CalcQuadTAB(RqArr);
end;
// ===========================================================================
// 
// ===========================================================================
end.
