unit AnimeGif01;

interface
uses
  //  
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  ActiveX, Dialogs, ExtCtrls,
  //  GDI+
  GDIPAPI, GDIPOBJ;
// ==========================================================================
//  21.06.2014
//     "    GIF"
type TonCountDown = procedure (const LoopCount : integer) of object;

// 15.06.2014
type TTimeArr  = array[0..65535] of integer;
     TPTimeArr = ^TTimeArr;

// 15.06.2014     
type TAnimeGif = class(TObject)
  private
    //--------------------------------
    fApplication : TApplication;
    //--------------------------------
    fFileName    : string;            //   
    fGDbmp       : TGPBitmap;         // GPBitmap  *.GIF
    fGDw, fGDh   : cardinal;          //   GPBitmap
    fFrCount     : cardinal;          //     GIF
    fPrCount     : cardinal;          //  property   GIF
    fLoopCount   : word;              //   GIF 
    //--------------------------------
    fFdBufSize   : cardinal;          //     FrameDelay
    fPFdBuf      : PPropertyItem;     // pointer   FrameDelay
    fPTimeArr    : TPTimeArr;         // pointer    FrameDelay
    fDelayGif    : integer;           // Gif    
    //--------------------------------
    fDelayAdd  : integer;             // Add    
    fIncrm     : boolean;             //    
    fZoom      : double;              //  
    fIMode     : TInterpolationMode;  //    
    fColor     : TColor;              //   Image
    //--------------------------------
    fCycle     : boolean;             //  -   
    fRun       : boolean;             //  -  
    //--------------------------------
    fonCountDown : TonCountDown;      //    CountDown
    //--------------------------------
    function  GetFrTime(Ind : integer) : integer;
    function  GetFrameDelay(RqFrameNum : integer) : integer;
    procedure FreeFrameDelay();
    function  CreateTimeArr() : boolean;
    procedure LookEventsAndSleep();
    function  GetLoopCount()  : boolean;
    function  ColorToTGPColor (RqColor : TColor) : TGPColor;
    //--------------------------------
  public
    //--------------------------------
    //  InterpolationMode   
    procedure   SetIMode(RqModeIndex : integer);
    //  GIF   GDI+ BitMap
    function    LoadGifFile (RqOpenDialog : TOpenDialog) : boolean;
    //  GIF  RqH, RqW   RqImg
    procedure   ShowScaledGif(RqImg : TImage; RqH, RqW  : cardinal);
    //  fZoom  GIF   RqImg
    procedure   ShowZoomGif(RqImg : TImage);
    //     
    // ,      RqTrW, RqTrH
    procedure   ShowThumbnailImage(RqImg : TImage; RqTrW, RqTrH : integer);
    //  
    constructor Create(Application : TApplication);
    //  
    procedure   Free();
    //--------------------------------
    //     GDI+ BitMap
    property GDbmp    : TGPBitmap read fGDbmp;
    //    GDI+ BitMap
    property GDw      : cardinal  read fGDw;
    //    GDI+ BitMap
    property GDh      : cardinal  read fGDh;
    //     GDI+ BitMap
    property FrCount  : cardinal  read fFrCount;
    //    GIF 
    property LoopCount  : word    read fLoopCount;
    //   (msec)     Ind
    property FrTime[Ind : integer] : integer   read  GetFrTime;
    //       
    property Cycle    : boolean   read fCycle  write fCycle;
    //       
    property Run      : boolean   read fRun    write fRun;
    //     
    property Zoom     : double    read fZoom   write fZoom;
    //        
    property DelayAdd : integer   read fDelayAdd write fDelayAdd;
    //       
    property Incrm    : boolean   read fIncrm   write fIncrm;
    //    
    property IMode    : TInterpolationMode  read fIMode;
    //   Image
    property Color    : TColor read fColor write fColor;
    //     "    GIF"
    property onCountDown : TonCountDown read fonCountDown write fonCountDown;
end;

// ==========================================================================
implementation
// ==========================================================================
// --------------------------------------------------------------------------
// 15.06.2014
constructor TAnimeGif.Create(Application : TApplication);
begin
  inherited Create();
  fApplication := Application;
  // -------------------------------------
  fFdBufSize   := 0;
  fPFdBuf    := nil;
  // -------------------------------------
  fIMode       := InterpolationModeDefault;
  fZoom        := 1;
  fDelayAdd    := 0;
  fIncrm       := False;
  fCycle       := False;
  fRun         := False;
  fColor       := clBtnFace;
  // -------------------------------------
  fonCountDown := nil;
end;

// --------------------------------------------------------------------------
// 15.06.2014
procedure TAnimeGif.Free();
begin
  FreeFrameDelay();
  if Assigned (fGDbmp) then fGDbmp.Free;
  inherited Free();
end;

// --------------------------------------------------------------------------
// 15.06.2014
//  InterpolationMode   
procedure TAnimeGif.SetIMode(RqModeIndex : integer);
begin
  case RqModeIndex of
  0 : fIMode := InterpolationModeDefault;
  1 : fIMode := InterpolationModeLowQuality;
  2 : fIMode := InterpolationModeHighQuality;
  3 : fIMode := InterpolationModeBilinear;
  4 : fIMode := InterpolationModeBicubic;
  5 : fIMode := InterpolationModeNearestNeighbor;
  6 : fIMode := InterpolationModeHighQualityBilinear;
  7 : fIMode := InterpolationModeHighQualityBicubic;
  else fIMode := InterpolationModeDefault;
  end;
end;

// --------------------------------------------------------------------------
// 15.06.2014
//  property FrTime
function  TAnimeGif.GetFrTime(Ind : integer) : integer;
begin
   Result := 0;
   if Ind < 0 then Exit;
   if (fFdBufSize > 0)   and
      (fPTimeArr <> nil) and
      (Cardinal(Ind) < fFrCount)   and
      (Ind >= 0)
    then Result := (fPTimeArr^[Ind] * 10);
end;

// --------------------------------------------------------------------------
// 15.06.2014
//       
procedure TAnimeGif.LookEventsAndSleep();
const TimeStep  = 200;                    //    
var   Delay     : integer;
      StepCount : integer;
begin
  Delay := (fDelayGif * 10) + fDelayAdd;
  if Delay <= 0
  then begin
    fApplication.ProcessMessages;
    Sleep(1);
  end
  else begin
    StepCount := Delay div TimeStep;
    while StepCount > 0 do
    begin
      fApplication.ProcessMessages;
      Sleep(TimeStep);
      Delay := Delay - TimeStep;
      StepCount := StepCount - 1;
    end;
    if Delay > 0
    then begin
       fApplication.ProcessMessages;
       Sleep(Delay);
    end;
  end;
end;

// --------------------------------------------------------------------------

function TAnimeGif.ColorToTGPColor (RqColor : TColor) : TGPColor;
type TRGB = record
     RI : byte;  //  R  nIndex  GetSysColor
     G  : byte;  //  G
     B  : byte;  //  B
     S  : byte;  //  $FF  SysColor
end;
var  PC : ^TRGB;
     SC : TRGB;
begin
    PC := Addr(RqColor);
    if PC^.S = 0
    then Result := MakeColor(255, PC^.RI, PC^.G, PC^.B)
    else begin
      SC := TRGB(GetSysColor(Integer(PC^.RI)));
      Result := MakeColor(255, SC.RI, SC.G, SC.B);
    end;
end;

// --------------------------------------------------------------------------
// 15.06.2014
//  GIF  RqH, RqW   RqImg
procedure TAnimeGif.ShowScaledGif(RqImg     : TImage;
                                  RqH, RqW  : cardinal);
var WGraph        : TGPGraphics;
    WBrush        : TGPSolidBrush;
    GPColor       : TGPColor;         //   Image
    LoopCountDown : integer;
    Ind           : integer;
begin
   if not Assigned(fGDbmp) then Exit;

   //  TColor  TGPColor
   GPColor := ColorToTGPColor(fColor);
   //   TGPSolidBrush
   WBrush := TGPSolidBrush.Create(GPColor);

   //   Bitmap
   with RqImg.Picture.Bitmap
   do begin
      PixelFormat := pf24bit;
      Height := RqH;
      Width  := RqW;
   end;
   LoopCountDown := fLoopCount;
   //    " "
   if Assigned(fonCountDown) then fonCountDown(LoopCountDown);
   //  WGraph   RqImg
   WGraph := TGPGraphics.Create(RqImg.Picture.Bitmap.Canvas.Handle);
   WGraph.SetInterpolationMode(fIMode);
   //   GIF   RqImg
   if fFrCount > 0
   then begin
     repeat
        for Ind := 0 to fFrCount - 1 do
        begin
          //  RqImg
          if not fIncrm
          then WGraph.FillRectangle(WBrush, 0,0, RqW, RqH);
          //    ()
          fGDbmp.SelectActiveFrame(FrameDimensionTime, Ind);
          // DrawImage to RqImg
          WGraph.DrawImage(fGDbmp, 0,0, RqW, RqH);
          //   
          RqImg.Repaint;
          //     
          fDelayGif := GetFrameDelay(Ind);
          //        
          LookEventsAndSleep();
          //     GIF
          if not fRun then Break;
        end;
        //     GIF
        if (not fCycle)
        then begin
            //    
            LoopCountDown := LoopCountDown - 1;
            if LoopCountDown < 0 then Break
            else if Assigned(fonCountDown)
                 then fonCountDown(LoopCountDown);
        end;
     until (not fRun);
   end;
   //  
   WBrush.Free;
   WGraph.Free;
end;

// --------------------------------------------------------------------------
// 15.06.2014
//  fZoom  GIF   RqImg
procedure TAnimeGif.ShowZoomGif(RqImg : TImage);
var ZmH, ZmW  : cardinal;
begin
   ZmH  := Round(fGDh * fZoom);
   ZmW  := Round(fGDw * fZoom);
   ShowScaledGif(RqImg, ZmH, ZmW);
end;

// --------------------------------------------------------------------------
// 15.06.2014
//     
// ,      RqTrW, RqTrH
procedure TAnimeGif.ShowThumbnailImage(RqImg : TImage;
                                       RqTrW, RqTrH : integer);
var WGraph   : TGPGraphics;
    TrW, TrH : cardinal;
    WCount   : cardinal;
begin
   if not Assigned(fGDbmp)     then Exit;
   if (fGDw < 1) or (fGDh < 1) then Exit;
   //   Image  
   //     RqTrW, RqTrH
   TrW := RqTrW;
   TrH := Round(fGDh * TrW / fGDw);
   if  (TrH > cardinal(RqTrH))
   then begin
      TrH := cardinal(RqTrH);                          
      TrW := Round(fGDw * TrH / fGDh);
   end;
   //  Bitmap   Image
   with RqImg.Picture.Bitmap
   do begin
     PixelFormat := pf24bit;
     Height := TrH;
     Width  := TrW;
   end;
   //     
   WCount := fGDbmp.GetFrameCount(FrameDimensionTime);
   if WCount > 0
   then begin
      fGDbmp.SelectActiveFrame(FrameDimensionTime, 0);
      WGraph := TGPGraphics.Create(RqImg.Picture.Bitmap.Canvas.Handle);
      WGraph.SetInterpolationMode(InterpolationModeBicubic);
      WGraph.DrawImage(fGDbmp, 0,0, TrW, TrH);
      WGraph.Free;
   end;
end;

// --------------------------------------------------------------------------
// 15.06.2014
//        RqFrameNum
function TAnimeGif.GetFrameDelay(RqFrameNum : integer) : integer;
begin
    if (fFdBufSize > 0) and (fPTimeArr <> nil)
    then Result := fPTimeArr^[RqFrameNum]
    else Result := 0;
end;

// --------------------------------------------------------------------------
// 15.06.2014
//    
procedure TAnimeGif.FreeFrameDelay();
begin
   if (fFdBufSize > 0) and (fPFdBuf <> nil)
   then begin
      try
         FreeMem(fPFdBuf, fFdBufSize);
      finally
         fPFdBuf    := nil;    // pointer   FrameDelay
         fFdBufSize := 0;      //     FrameDelay
      end;
   end;
end;

// --------------------------------------------------------------------------
// 15.06.2014
//    
function TAnimeGif.CreateTimeArr() : boolean;
var RqItem    : PropertyItem;
begin
   Result     := False;
   fPTimeArr  := nil;
   //    FrameDelay
   FreeFrameDelay();
   //   
   fPrCount := fGDbmp.GetPropertyCount;
   if (fPrCount > 0) and (fGDbmp.GetLastStatus = Ok)
   then begin
      //
      fFdBufSize := fGDbmp.GetPropertyItemSize(PropertyTagFrameDelay);
      if (fFdBufSize > 0) and (fGDbmp.GetLastStatus = Ok)
      then begin
        try
           //     
           GetMem(fPFdBuf, fFdBufSize);
        except
           fPFdBuf    := nil;
           fFdBufSize := 0;
        end;
        //  
        if fPFdBuf <> nil
        then begin
           fGDbmp.GetPropertyItem(PropertyTagFrameDelay, fFdBufSize, fPFdBuf);
           if (fGDbmp.GetLastStatus = Ok)
           then begin
              //     PropertyItem
              RqItem := fPFdBuf^;
              if (RqItem.value <> nil) and
                 (RqItem.type_ = PropertyTagTypeLong)
              then begin
                 //     
                 //    
                 fPTimeArr := TPTimeArr(RqItem.value);
                 Result := True;
              end;
           end;
        end;
      end;
   end;
end;

// --------------------------------------------------------------------------
// 21.06.2014
// {$OPTIMIZATION OFF}
//     
function TAnimeGif.GetLoopCount() : boolean;
var  Item    : PropertyItem;
     BufSize   : cardinal;
     PBUF      : PPropertyItem;
     pValue    : ^word;
begin
   Result      := False;
   fLoopCount  := 0;
   PBUF        := nil;
   //   
   fPrCount := fGDbmp.GetPropertyCount;
   if (fPrCount > 0) and (fGDbmp.GetLastStatus = Ok)
   then begin
      BufSize := fGDbmp.GetPropertyItemSize(PropertyTagLoopCount);
      if BufSize < 1 then Exit;
      try
        GetMem(PBUF, BufSize);
        fGDbmp.GetPropertyItem(PropertyTagLoopCount, BufSize, PBUF);
        if (fGDbmp.GetLastStatus = Ok)
        then begin
            //     PropertyItem
            Item := PBUF^;
            if (Item.value <> nil) and (Item.type_ = PropertyTagTypeShort)
            then begin
               //    
               pValue := Item.value;
               fLoopCount := pValue^;
               Result := True;
            end;
        end;
       finally
         if (PBUF <> nil) and (BufSize > 0)
         then FreeMem(PBUF, BufSize);
       end;
   end;
end;
// {$OPTIMIZATION ON}

// --------------------------------------------------------------------------
// 15.06.2014
//  GIF   GDI+ BitMap
function TAnimeGif.LoadGifFile(RqOpenDialog : TOpenDialog) : boolean;
begin
 Result := False;
 if RqOpenDialog.Execute
 then begin
    fFileName := RqOpenDialog.FileName;
    if Assigned(fGDbmp)
    then begin
       fGDbmp.Free;
       fGDbmp := nil;
    end;
    try
       //      GDbmp
       fGDbmp := TGPBitmap.Create(fFileName);
       //   TGPBitmap
       fGDw := fGDbmp.GetWidth;
       fGDh := fGDbmp.GetHeight;
       fFrCount := fGDbmp.GetFrameCount(FrameDimensionTime);
       if fFrCount > 0
       then begin
          GetLoopCount();
          CreateTimeArr();
          Result := True;
       end;
    except
       fFileName := '';
    end;
 end;
end;

// ==========================================================================
// ==========================================================================
end.
